using System;
using UnityEngine;
using UnityEngine.XR.ARFoundation;
using UnityEngine.XR.ARSubsystems;

namespace Modules.MachineRecognition.Scripts
{
    /// <summary>
    /// Represents information about a machine.
    /// </summary>
    [Serializable]
    public class MachineInfo
    {
        /// <summary>
        /// The unique identifier of the machine.
        /// </summary>
        public int id;

        /// <summary>
        /// The name of the machine.
        /// </summary>
        public string name;

        /// <summary>
        /// A brief description of the machine.
        /// </summary>
        public string description;
    }

    /// <summary>
    /// Controls the behavior of a tracked image in an AR environment.
    /// Handles initialization, positioning, rotation, and interaction with a panel manager.
    /// </summary>
    public class TrackedImageController : MonoBehaviour
    {
        /// <summary>
        /// A reference to the PanelManager component associated with the tracked image.
        /// Used to manage the UI panel and handle data updates related to the tracked image.
        /// </summary>
        private PanelManager _panel;

        /// <summary>
        /// A reference to the HTTPClient component used for making HTTP requests.
        /// </summary>
        private HTTPClient _httpClient;

        /// <summary>
        /// The reference image associated with the tracked image.
        /// </summary>
        private XRReferenceImage _referenceImage;

        /// <summary>
        /// Indicates whether the controller has been initialized.
        /// </summary>
        private bool _isInitialized;

        /// <summary>
        /// Initializes the tracked image controller with the provided ARTrackedImage.
        /// Sets up necessary components, updates the tracked image, and fetches machine information.
        /// </summary>
        /// <param name="trackedImage">The ARTrackedImage to initialize the controller with.</param>
        public void Initialize(ARTrackedImage trackedImage)
        {
            if (_isInitialized) return;

            _httpClient = GetComponent<HTTPClient>();
            _panel = GetComponent<PanelManager>();

            SetTrackedImage(trackedImage);

            if (int.TryParse(_referenceImage.name, out var imageId))
            {
                if (imageId == 0) SetFuzzyPanel();
                SetPanelUrl(imageId);
                GetMachineInfo(imageId);
            }

            _isInitialized = true;
        }

        /// <summary>
        /// Parses machine information from a JSON string and updates the panel with the machine's details.
        /// </summary>
        /// <param name="data">The JSON string containing machine information.</param>
        private void SetMachineInfo(string data)
        {
            var machineInfo = JsonUtility.FromJson<MachineInfo>(data);
            _panel.UpdateMachineInfo(machineInfo.id, machineInfo.name, machineInfo.description);
        }

        /// <summary>
        /// Sends a GET request to retrieve machine information based on the image ID.
        /// The response is processed to update the panel with the machine's details.
        /// </summary>
        /// <param name="imageId">The unique identifier of the image used to fetch machine information.</param>
        private void GetMachineInfo(int imageId)
        {
            var url = $"https://api.arassistant.nl/machines/{imageId}";
            StartCoroutine(_httpClient.GetRequest(url, SetMachineInfo));
        }

        /// <summary>
        /// Configures the panel to handle fuzzy logic data and adjusts the UI accordingly.
        /// Disables the sensor graph panel and repositions the data panel.
        /// </summary>
        private void SetFuzzyPanel()
        {
            _panel.SetDataType(DataType.FuzzyLogicData);

            GetComponentInChildren<SensorGraphController>()?.gameObject.SetActive(false);

            var dataPanel = GameObject.Find("DataPanel");
            if (dataPanel != null) dataPanel.transform.localPosition = new Vector3(0f, dataPanel.transform.localPosition.y, dataPanel.transform.localPosition.z);
        }

        /// <summary>
        /// Sets the tracked image and updates the position and rotation of the prefab.
        /// </summary>
        /// <param name="trackedImage">The ARTrackedImage to set.</param>
        public void SetTrackedImage(ARTrackedImage trackedImage)
        {
            _referenceImage = trackedImage.referenceImage;

            Vector3 trackedPosition = trackedImage.transform.position;
            float cameraHeight = Camera.main != null ? Camera.main.transform.position.y - 0.15f : trackedPosition.y;

            transform.position = new Vector3(trackedPosition.x, cameraHeight, trackedPosition.z);

            SetPrefabRotation(trackedImage);
        }


        /// <summary>
        /// Adjusts the rotation of the prefab based on the tracked image and camera position.
        /// Ensures the prefab faces the camera or adjusts its rotation relative to the image.
        /// </summary>
        /// <param name="trackedImage">The ARTrackedImage to use for rotation calculations.</param>
        private void SetPrefabRotation(ARTrackedImage trackedImage)
        {
            if (Camera.main == null)
            {
                transform.rotation = Quaternion.Euler(0f, trackedImage.transform.eulerAngles.y, 0f);
                return;
            }

            var directionToCamera = Camera.main.transform.position - trackedImage.transform.position;
            directionToCamera.y = 0;

            if (directionToCamera.sqrMagnitude == 0) return;

            transform.rotation = Quaternion.LookRotation(directionToCamera) * Quaternion.Euler(0f, -180f, 0f);
        }

        /// <summary>
        /// Sets the URL for the panel manager based on the image ID and starts listening to SSE events.
        /// </summary>
        /// <param name="imageId">The ID of the image used to construct the URL.</param>
        private void SetPanelUrl(int imageId)
        {
            _panel.SetUrl(GetCorrectPanelUrl(imageId));
        }

        /// <summary>
        /// Constructs the correct URL for the panel manager based on the image ID and data type.
        /// </summary>
        /// <param name="imageId">The ID of the image used to construct the URL.</param>
        /// <returns>
        /// A string representing the URL for the panel manager. The URL varies depending on the data type:
        /// <list type="bullet">
        /// <item><see cref="DataType.SensorData"/>: Returns a URL for MQTT data streams, including the image ID.</item>
        /// <item><see cref="DataType.FuzzyLogicData"/>: Returns a URL for maintenance evaluation.</item>
        /// <item>Default: Returns a fallback MQTT URL with a default image ID of 1.</item>
        /// </list>
        /// </returns>
        private string GetCorrectPanelUrl(int imageId)
        {
            switch (_panel.dataType)
            {
                case DataType.SensorData:
                    return $"https://api.arassistant.nl/mqtt/{imageId}/%23";
                case DataType.FuzzyLogicData:
                    return "https://api.arassistant.nl/maintenance/evaluate";
                default:
                    return "https://api.arassistant.nl/mqtt/1/#";
            }
        }

        /// <summary>
        /// Hides the tracked image prefab and stops the panel manager from listening to SSE events.
        /// </summary>
        public void Hide()
        {
            if (!gameObject.activeSelf) return;

            if (_panel == null) return;

            _panel.SetUrl(string.Empty);
            _panel.StopListeningToSse();

            gameObject.SetActive(false);
        }
    }
}